!===============================================================================
! Copyright (C) 2025 Intel Corporation
!
! This software and the related documents are Intel copyrighted  materials,  and
! your use of  them is  governed by the  express license  under which  they were
! provided to you (License).  Unless the License provides otherwise, you may not
! use, modify, copy, publish, distribute,  disclose or transmit this software or
! the related documents without Intel's prior written permission.
!
! This software and the related documents  are provided as  is,  with no express
! or implied  warranties,  other  than those  that are  expressly stated  in the
! License.
!===============================================================================

!
!   Content : Intel(R) oneAPI Math Kernel Library (oneMKL) IE Sparse BLAS
!             Fortran example for mkl_sparse_convert_{csc, coo} functions and
!             for mkl_sparse_d_export_{csc, coo} functions
!
!*******************************************************************************
!
! Example program for using Intel oneMKL Inspector-Executor Sparse BLAS routines
! for conversions between different matrix formats.
!
! The following Inspector Executor Sparse Blas routines are used in the example:
!
!   Initialization/Destruction stage:
!          mkl_sparse_d_create_csr
!          mkl_sparse_destroy
!
!   Format conversion functions:
!          mkl_sparse_convert_csc   mkl_sparse_convert_coo
!
!   Sparse matrix export functions:
!          mkl_sparse_d_export_csc  mkl_sparse_d_export_coo
!
!   Sparse matrix-vector multiplication function:
!          mkl_sparse_d_mv
!
!   Sparse matrix ordering function:
!          mkl_sparse_order
!
! Consider the matrix A below to be represented in multiple sparse formats
! (see 'Sparse Matrix Storage Schemes' in the Intel oneMKL Reference Manual):
!
!       |  1  -2   0   0 |
!       |  3  -4   0   0 |
!   A = |  0   0   5  -6 |.
!       |  0   0   7  -8 |
!       |  9 -10   0   0 |
!       | 11 -12   0   0 |
!
!  A compressed sparse row (CSR) representation of the matrix with
!  three arrays is:
!
!     csrNrows  = 6
!     csrNcols  = 4
!     csrNnz    = 12
!     csrIndex  = SPARSE_INDEX_BASE_ZERO
!     csrRowPtr = (0       2       4       6       8      10      12)
!     csrColIdx = (0   1   0   1   2   3   2   3   0   1   0   1)
!     csrValues = (1  -2   3  -4   5  -6   7  -8   9 -10  11 -12)
!
!  A coordinate format (COO) representation of the matrix is:
!
!     cooNrows  = 6
!     cooNcols  = 4
!     cooNnz    = 12
!     cooIndex  = SPARSE_INDEX_BASE_ZERO
!     cooRowIdx = (0   0   1   1   2   2   3   3   4   4   5   5)
!     cooColIdx = (0   1   0   1   2   3   2   3   0   1   0   1)
!     cooValues = (1  -2   3  -4   5  -6   7  -8   9 -10  11 -12)
!
!  A compressed sparse column (CSC) representation of the matrix with
!  three arrays is:
!
!     cscNrows  = 6
!     cscNcols  = 4
!     cscNnz    = 12
!     cscIndex  = SPARSE_INDEX_BASE_ZERO
!     cscColPtr = (0               4               8      10      12)
!     cscRowIdx = (0   1   4   5   0   1   4   5   2   3   2   3)
!     cscValues = (1   3   9  11  -2  -4 -10 -12   5   7  -6  -8)
!
!  This example presents:
!    * mkl_sparse_convert_{csc/coo}() usage to convert a sparse matrix:
!      * from CSR to CSC,
!      * from CSR to COO,
!    8 mkl_sparse_d_export_{csc/coo}() usage to extract CSC and COO arrays,
!
!  Note that we will call mkl_sparse_order() after conversion to CSC so that when printing out
!  we have it printed in sorted order in this example.  This is not strictly necessary as
!  most algorithms work with unsorted data (exception is unoptimized TRSV/M which requires lower and
!  upper parts to be separated within each row/column for CSR/CSC/BSR formats.  Algorithms can
!  sometimes be more efficient with sorted data due to better memory access patterns.
!
!*******************************************************************************

SUBROUTINE print_int_value(name, val)
    IMPLICIT NONE
    CHARACTER(LEN=*), INTENT(IN) :: name
    INTEGER, INTENT(IN) :: val
    PRINT *, TRIM(ADJUSTL(name)), " = ", val
END SUBROUTINE print_int_value

SUBROUTINE print_int_array(name, array, len)
    IMPLICIT NONE
    CHARACTER(LEN=*), INTENT(IN) :: name
    INTEGER, INTENT(IN), DIMENSION(*) :: array
    INTEGER, INTENT(IN) :: len
    INTEGER :: i
    WRITE(*, '(A, A, A)', ADVANCE="NO") " ", TRIM(ADJUSTL(name)), " ="
    DO i = 1, len
        WRITE(*,  '(I3)', ADVANCE="NO") array(i)
        IF (i < len) WRITE(*, '(A)', ADVANCE="NO") ", "
    END DO
    PRINT *
END SUBROUTINE print_int_array

SUBROUTINE print_index(name, idxBase)
    USE MKL_SPBLAS ! For SPARSE_INDEX_BASE_ZERO
    IMPLICIT NONE
    CHARACTER(LEN=*), INTENT(IN) :: name
    INTEGER(C_INT), INTENT(IN) :: idxBase
    CHARACTER(LEN=25) :: idxBaseStr
    IF (idxBase == SPARSE_INDEX_BASE_ZERO) THEN
        idxBaseStr = "SPARSE_INDEX_BASE_ZERO"
    ELSE
        idxBaseStr = "SPARSE_INDEX_BASE_ONE"
    END IF
    PRINT *, TRIM(ADJUSTL(name)), " = ", TRIM(idxBaseStr)
END SUBROUTINE print_index

SUBROUTINE print_flt_array(name, array, len)
    IMPLICIT NONE
    CHARACTER(LEN=*), INTENT(IN) :: name
    DOUBLE PRECISION, INTENT(IN), DIMENSION(*) :: array
    INTEGER, INTENT(IN) :: len
    INTEGER :: i
    WRITE(*, '(A, A, A)', ADVANCE="NO") " ", TRIM(ADJUSTL(name)), " ="
    DO i = 1, len
        WRITE(*,  '(F4.0)', ADVANCE="NO") array(i)
        IF (i < len) WRITE(*, '(A)', ADVANCE="NO") ", "
    END DO
    PRINT *
END SUBROUTINE print_flt_array

PROGRAM SPARSE_CONVERT_COO_AND_CSC
    USE MKL_SPBLAS
    USE ISO_C_BINDING
    IMPLICIT NONE

    INTEGER, PARAMETER :: M = 6, N = 4, NNZ = 12
    INTEGER :: csrNrows, csrNcols, csrNnz
    INTEGER :: cooNrows, cooNcols, cooNnz
    INTEGER :: cscNrows, cscNcols, cscNnz
    INTEGER(C_INT) cscIdxBase, cooIdxBase
    INTEGER, ALLOCATABLE :: csrRowPtr(:), csrColIdx(:)
    DOUBLE PRECISION, ALLOCATABLE :: csrValues(:)
    TYPE(C_PTR)      :: cscColStart_c   , cscColEnd_c   , cscRowIdx_c
    INTEGER, POINTER :: cscColStart_f(:), cscColEnd_f(:), cscRowIdx_f(:)
    TYPE(C_PTR)      :: cooRowIdx_c   , cooColIdx_c
    INTEGER, POINTER :: cooRowIdx_f(:), cooColIdx_f(:)
    TYPE(C_PTR)               :: cscValues_c   , cooValues_c
    DOUBLE PRECISION, POINTER :: cscValues_f(:), cooValues_f(:)
    DOUBLE PRECISION :: x_N(N), y_M(M)
    DOUBLE PRECISION csrSum, cscSum, cooSum
    TYPE(SPARSE_MATRIX_T) :: csrA, cscA, cooA
    TYPE(MATRIX_DESCR) :: descrA
    INTEGER :: info, exit_info
    LOGICAL :: csc_passed, coo_passed
    DOUBLE PRECISION :: alpha, beta
    DOUBLE PRECISION, PARAMETER :: TOL = 1.0e-6

    ! Initialize variables
    csrNrows = M
    csrNcols = N
    csrNnz = NNZ
    ALLOCATE(csrRowPtr(M+1))
    ALLOCATE(csrColIdx(NNZ))
    ALLOCATE(csrValues(NNZ))
    csrRowPtr = (/ 0, 2, 4, 6, 8, 10, 12 /)
    csrColIdx = (/ 0, 1, 0, 1, 2, 3, 2, 3, 0, 1, 0, 1 /)
    csrValues = (/ 1.0, -2.0, 3.0, -4.0, 5.0, -6.0, 7.0, -8.0, 9.0, -10.0, 11.0, -12.0 /)
    x_N = (/ 1.0, 1.0, 1.0, 1.0/)
    y_M = (/ 0.0, 0.0, 0.0, 0.0, 0.0, 0.0/)
    csrSum = 0.0
    cscSum = 0.0
    cooSum = 0.0
    csc_passed = .false.
    coo_passed = .false.
    descrA%type = SPARSE_MATRIX_TYPE_GENERAL
    alpha = 1.0
    beta = 0.0

    print *, "EXAMPLE PROGRAM for matrix format conversion routines from IE Sparse BLAS"
    print *, "-------------------------------------------------------------------------"

    ! Create CSR sparse matrix handle
    info = mkl_sparse_d_create_csr(csrA, SPARSE_INDEX_BASE_ZERO, csrNrows, csrNcols, csrRowPtr, csrRowPtr(2:), csrColIdx, csrValues)
    if (info /= SPARSE_STATUS_SUCCESS) then
        print *, "Error in mkl_sparse_d_create_csr: ", info
        exit_info = 1
        stop
    end if

    ! Compute CSR sum
    csrSum = sum(csrValues)

    ! Print input CSR matrix
    print *, "[Input] Matrix arrays in CSR format:"
    call print_int_value("nrows", csrNrows)
    call print_int_value("ncols", csrNcols)
    call print_int_value("nnz  ", csrNnz)
    call print_index("index", SPARSE_INDEX_BASE_ZERO)
    call print_int_array("csrRowPtr", csrRowPtr, csrNrows+1)
    call print_int_array("csrColIdx", csrColIdx, csrNnz)
    call print_flt_array("csrValues", csrValues, csrNnz)

    ! Convert from CSR to COO format
    info = mkl_sparse_convert_coo(csrA, SPARSE_OPERATION_NON_TRANSPOSE, cooA)
    if (info /= SPARSE_STATUS_SUCCESS) then
        print *, "Error in mkl_sparse_convert_coo: ", info
        exit_info = 1
        stop
    end if

    ! Export COO matrix
    info = mkl_sparse_d_export_coo(cooA, cooIdxBase, cooNrows, cooNcols, cooNnz, cooRowIdx_c, cooColIdx_c, cooValues_c)
    if (info /= SPARSE_STATUS_SUCCESS) then
        print *, "Error in mkl_sparse_d_export_coo: ", info
        exit_info = 1
        stop
    end if

    !   Converting C into Fortran pointers
    call C_F_POINTER(cooRowIdx_c, cooRowIdx_f, [cooNnz])
    call C_F_POINTER(cooColIdx_c, cooColIdx_f, [cooNnz])
    call C_F_POINTER(cooValues_c, cooValues_f, [cooNnz])

    ! Compute COO sum
    info = mkl_sparse_d_mv(SPARSE_OPERATION_NON_TRANSPOSE, alpha, cooA, descrA, x_N, beta, y_M)
    cooSum = sum(y_M)
    if (abs(csrSum - cooSum) < TOL) coo_passed = .true.

    ! Print output COO matrix
    print *, "[Output] Matrix arrays in COO format:"
    call print_int_value("nrows", cooNrows)
    call print_int_value("ncols", cooNcols)
    call print_int_value("nnz  ", cooNnz)
    call print_index("index", cooIdxBase)
    call print_int_array("cooRowIdx", cooRowIdx_f, cooNnz)
    call print_int_array("cooColIdx", cooColIdx_f, cooNnz)
    call print_flt_array("cooValues", cooValues_f, cooNnz)
    if (coo_passed) then
        print *, "COO Test Passed"
    else
        print *, "COO Test Failed: CSR sum =", csrSum, "but COO sum =", cooSum
    end if

    ! Convert from CSR to CSC format
    info = mkl_sparse_convert_csc(csrA, SPARSE_OPERATION_NON_TRANSPOSE, cscA)
    if (info /= SPARSE_STATUS_SUCCESS) then
        print *, "Error in mkl_sparse_convert_csc: ", info
        exit_info = 1
        stop
    end if

    ! Export CSC matrix
    info = mkl_sparse_d_export_csc(cscA, cscIdxBase, cscNrows, cscNcols, cscColStart_c, cscColEnd_c, cscRowIdx_c, cscValues_c)
    if (info /= SPARSE_STATUS_SUCCESS) then
        print *, "Error in mkl_sparse_d_export_csc: ", info
        exit_info = 1
        stop
    end if

    !   Converting C into Fortran pointers
    call C_F_POINTER(cscColStart_c, cscColStart_f, [cscNcols])
    call C_F_POINTER(cscColEnd_c, cscColEnd_f, [cscNcols])
    cscNnz = cscColEnd_f(cscNcols)
    call C_F_POINTER(cscRowIdx_c, cscRowIdx_f, [cscNnz])
    call C_F_POINTER(cscValues_c, cscValues_f, [cscNnz])

    ! Compute CSC sum
    info = mkl_sparse_d_mv(SPARSE_OPERATION_NON_TRANSPOSE, alpha, cscA, descrA, x_N, beta, y_M)
    if (info /= SPARSE_STATUS_SUCCESS) then
        print *, "Error in mkl_sparse_d_mv: ", info
        exit_info = 1
        stop
    end if
    cscSum = sum(y_M)
    if (abs(csrSum - cscSum) < TOL) csc_passed = .true.

    ! Print output CSC matrix
    print *, "[Output] Matrix arrays in CSC format:"
    call print_int_value("nrows", cscNrows)
    call print_int_value("ncols", cscNcols)
    call print_int_value("nnz  ", cscNnz)
    call print_index("index", cscIdxBase)
    call print_int_array("cscColPtr", cscColStart_f, cscNcols+1)
    call print_int_array("cscRowIdx", cscRowIdx_f, cscNnz)
    call print_flt_array("cscValues", cscValues_f, cscNnz)
    if (csc_passed) then
        print *, "CSC Test Passed"
    else
        print *, "CSC Test Failed: CSR sum =", csrSum, "but CSC sum =", cscSum
    end if

    ! Release matrix handles
    info = mkl_sparse_destroy(csrA)
    info = mkl_sparse_destroy(cooA)
    info = mkl_sparse_destroy(cscA)

    DEALLOCATE(csrRowPtr)
    DEALLOCATE(csrColIdx)
    DEALLOCATE(csrValues)

END PROGRAM SPARSE_CONVERT_COO_AND_CSC
